<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Support\ValidationRule;
use App\Models\Admin\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redirect;

class UserController extends Controller
{
    public function index(){
        $user = User::query()->get();

        $data = [
            'user' => $user
        ];
        return view('Admin.User.index', $data);
    }

    public function store(Request $request){
        $rule = [
            'name' => 'required|string',
            'email' => 'required|email:rfc,dns',
            'password' => 'required|regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).{8,}$/|min:8',
        ];

        $request->validate($rule, ValidationRule::getErrorMessage($rule));

        DB::beginTransaction();

        try{
            $user = [
                'name' => strip_tags($request->name),
                'email' => strip_tags($request->email),
                'password' => Hash::make(strip_tags($request->password)),
            ];

            User::query()->create($user);

            DB::commit();

            return Redirect::route('user.index')
                ->with('success', 'User berhasil ditambahkan');
        }
        catch(Exception $e){
            Log::error($e);
            DB::rollBack();

            return Redirect::route('user.index')
                ->with('error', 'User gagal ditambahkan');
        }
    }

    public function update(Request $request, $id){
        $rule = [
            'name' => 'required|string',
            'email' => 'required|email:rfc,dns',
            'passwordbaru' => 'required|regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).{8,}$/|min:8',
            'passwordlama' => 'required|string'
        ];

        $request->validate($rule, ValidationRule::getErrorMessage($rule));

        DB::beginTransaction();

        try{
            $id = decrypt($id);
            $passwordbaru = Hash::make(strip_tags($request->passwordbaru));
            $passwordlama = strip_tags($request->passwordlama);

            $oldpassword = User::where('id', $id)->first();

            if (Hash::check($passwordlama, $oldpassword->password)) {
                // Passwords match
                $user = [
                    'name' => strip_tags($request->name),
                    'email' => strip_tags($request->email),
                    'password' => $passwordbaru
                ];

                User::where('id', $id)->update($user);

                DB::commit();

                return Redirect::route('user.index')
                    ->with('success', 'User berhasil diupdate');
            } else {
                // Passwords do not match
                return Redirect::route('user.index')
                    ->with('error', 'Password lama yang dimasukkan salah');
            }
        }
        catch(Exception $e){
            Log::error($e);
            DB::rollBack();

            return Redirect::route('user.index')
                ->with('error', 'User gagal diupdate');
        }
    }

    public function destroy($id) {
        $id = decrypt($id);

        DB::beginTransaction();

        try{
            User::where('id', $id)->delete();

            DB::commit();

            return Redirect::route('user.index')
                ->with('success', 'User berhasil dihapus');
        }
        catch(Exception $e){
            Log::error($e);
            DB::rollBack();

            return Redirect::route('user.index')
                ->with('error', 'User gagal dihapus');
        }
    }
}
